function New-PowerLabSwitch {
	param(
		[Parameter()]
		[string]$SwitchName = 'PowerLab',

		[Parameter()]
		[string]$SwitchType = 'External'
	)

	if (-not (Get-VmSwitch -Name $SwitchName -SwitchType $SwitchType -ErrorAction SilentlyContinue)) {
		$null = New-VMSwitch -Name $SwitchName -SwitchType $SwitchType
	} else {
		Write-Verbose -Message "Przełącznik [$($SwitchName)] już istnieje."
	}
}

function New-PowerLabVm {
	param(
		[Parameter(Mandatory)]
		[string]$Name,

		[Parameter()]
		[string]$Path = 'C:\PowerLab\VMs',

		[Parameter()]
		[string]$Memory = 4GB,

		[Parameter()]
		[string]$Switch = 'PowerLab',

		[Parameter()]
		[ValidateRange(1, 2)]
		[int]$Generation = 2,

		[Parameter()]
		[switch]$PassThru
	)

	if (-not (Get-Vm -Name $Name -ErrorAction SilentlyContinue)) {
		$null = New-VM -Name $Name -Path $Path -MemoryStartupBytes $Memory -Switch $Switch -Generation $Generation
	} else {
		Write-Verbose -Message "Maszyna wirtualna o nazwie [$($Name)] już istnieje."
	}
	if ($PassThru.IsPresent) {
		Get-VM -Name $Name
	}
}

function New-PowerLabVhd {
	param
	(
		[Parameter(Mandatory)]
		[string]$Name,

		[Parameter()]
		[string]$AttachToVm,

		[Parameter()]
		[ValidateRange(512MB, 1TB)]
		[int64]$Size = 50GB,

		[Parameter()]
		[ValidateSet('Dynamic', 'Fixed')]
		[string]$Sizing = 'Dynamic',

		[Parameter()]
		[string]$Path = 'C:\PowerLab\VHDs'
	)

	$vhdxFileName = "$Name.vhdx"
	$vhdxFilePath = Join-Path -Path $Path -ChildPath "$Name.vhdx"

	### Sprawdza, czy dysk VHD już istnieje; jeżeli nie, tworzy go
	if (-not (Test-Path -Path $vhdxFilePath -PathType Leaf)) {
		$params = @{
			SizeBytes = $Size
			Path      = $vhdxFilePath
		}
		if ($Sizing -eq 'Dynamic') {
			$params.Dynamic = $true
		} elseif ($Sizing -eq 'Fixed') {
			$params.Fixed = $true
		}

		New-VHD @params
		Write-Verbose -Message "Utworzono nowy dysk VHD w katalogu [$($vhdxFilePath)]"
	}

	### Przyłącza istniejący lub nowo utworzony dysk VHD do maszyny wirtualnej.
	if ($PSBoundParameters.ContainsKey('AttachToVm')) {
		if (-not ($vm = Get-VM -Name $AttachToVm -ErrorAction SilentlyContinue)) {
			Write-Warning -Message "Maszyna wirtualna o nazwie [$($AttachToVm)] nie istnieje. Nie można przyłączyć dysku VHD."
		} elseif (-not ($vm | Get-VMHardDiskDrive | Where-Object { $_.Path -eq $vhdxFilePath })) {
			$vm | Add-VMHardDiskDrive -Path $vhdxFilePath
			Write-Verbose -Message "Dysk VHDX [$($vhdxFilePath)] został przyłączony do maszyny wirtualnej [$($AttachToVM)]."
		} else {
			Write-Verbose -Message "Dysk VHDX [$($vhdxFilePath)] jest już przyłączony do maszyny wirtualnej [$($AttachToVM)]."
		}
	}
}

function Install-PowerLabOperatingSystem {
	param
	(
		[Parameter(Mandatory)]
		[string]$VmName,

		[Parameter()]
		[string]$OperatingSystem = 'Server 2016',

		[Parameter()]
		[ValidateSet('ServerStandardCore')]
		[string]$OperatingSystemEdition = 'ServerStandardCore',

		[Parameter()]
		[string]$DiskSize = 40GB,

		[Parameter()]
		[string]$VhdFormat = 'VHDX',

		[Parameter()]
		[string]$VhdType = 'Dynamic',

		[Parameter()]
		[string]$VhdPartitionStyle = 'GPT',

		[Parameter()]
		[string]$VhdBaseFolderPath = 'C:\PowerLab\VHDs',

		[Parameter()]
		[string]$IsoBaseFolderPath = 'C:\PowerLab\ISOs',

		[Parameter()]
		[string]$VhdPath
	)
	
	## Zakładamy, że w folderze mamy plik <VMName>.xml zawierający odpowiedzi dla instalacji nienadzorowanej
	$answerFile = Get-Item -Path "$PSScriptRoot\$VMName.xml"

	## Wywołanie skryptu w konwencji dot-source. Ponieważ w skrypcie znajduje się definicja funkcji, wykonanie tej operacji spowoduje jej udostępnienie
	. "$PSScriptRoot\Convert-WindowsImage.ps1"

	## Tutaj możemy dodać obsługę wielu systemów operacyjnych, wybierając odpowiedni obraz ISO w zależności od wybranej wersji systemu operacyjnego
	switch ($OperatingSystem) {
		'Server 2016' {
			$isoFilePath = "$IsoBaseFolderPath\en_windows_server_2016_x64_dvd_9718492.iso"
		}
		default {
			throw "Nierozpoznany parametr: [$_]"
		}
	}

	$convertParams = @{
		SourcePath        = $isoFilePath
		SizeBytes         = $DiskSize
		Edition           = $OperatingSystemEdition
		VHDFormat         = $VhdFormat
		VHDType           = $VhdType
		VHDPartitionStyle = 'GPT'
		UnattendPath      = $answerFile.FullName
	}
	if ($PSBoundParameters.ContainsKey('VhdPath')) {
		$convertParams.VHDPath = $VhdPath
	} else {
		$convertParams.VHDPath = "$VhdBaseFolderPath\$VMName.vhdx"
	}

	Convert-WindowsImage @convertParams

	$vm = Get-Vm -Name $VmName
	if (($vm | Get-VMHardDiskDrive).Path -ne $convertParams.VHDPath) {
		$vm | Add-VMHardDiskDrive -Path $convertParams.VHDPath
	}	
	$bootOrder = ($vm | Get-VMFirmware).Bootorder
	if ($bootOrder[0].BootType -ne 'Drive') {
		$vm | Set-VMFirmware -FirstBootDevice $vm.HardDrives[0]
	}
}

function New-PowerLabActiveDirectoryTestObject {
	param(
		[Parameter(Mandatory)]
		[string]$SpreadsheetPath
	)

	$users = Import-Excel -Path $SpreadsheetPath -WorksheetName Users
	$groups = Import-Excel -Path $SpreadsheetPath -WorksheetName Groups

	$cred = Import-CliXml -Path 'C:\PowerLab\DomainCredential.xml'
	$dcSession = New-PSSession -VMName LABDC -Credential $cred

	$scriptBlock = {
		foreach ($group in $using:groups) {
			if (-not (Get-AdOrganizationalUnit -Filter "Name -eq '$($group.OUName)'")) {
				New-AdOrganizationalUnit -Name $group.OUName
			}
			if (-not (Get-AdGroup -Filter "Name -eq '$($group.GroupName)'")) {
				New-AdGroup -Name $group.GroupName -GroupScope $group.Type -Path "OU=$($group.OUName),DC=powerlab,DC=local"
			}
		}

		foreach ($user in $using:users) {
			if (-not (Get-AdOrganizationalUnit -Filter "Name -eq '$($user.OUName)'")) {
				New-AdOrganizationalUnit -Name $user.OUName
			}
			if (-not (Get-AdUser -Filter "Name -eq '$($user.UserName)'")) {
				New-AdUser -Name $user.UserName -Path "OU=$($user.OUName),DC=powerlab,DC=local"
			}
			if ($user.UserName -notin (Get-AdGroupMember -Identity $user.MemberOf).Name) {
				Add-AdGroupMember -Identity $user.MemberOf -Members $user.UserName
			}
		}
	}

	Invoke-Command -Session $dcSession -ScriptBlock $scriptBlock
	$dcSession | Remove-PSSession
}

function New-PowerLabActiveDirectoryForest {
	param(
		[Parameter(Mandatory)]
		[pscredential]$Credential,

		[Parameter(Mandatory)]
		[string]$SafeModePassword,

		[Parameter()]
		[string]$VMName = 'LABDC',

		[Parameter()]
		[string]$DomainName = 'powerlab.local',

		[Parameter()]
		[string]$DomainMode = 'WinThreshold',

		[Parameter()]
		[string]$ForestMode = 'WinThreshold'
	)

	Invoke-Command -VMName $VMName -Credential $Credential -ScriptBlock {

		Install-windowsfeature -Name AD-Domain-Services
		
		$forestParams = @{
			DomainName                    = $using:DomainName
			DomainMode                    = $using:DomainMode
			ForestMode                    = $using:ForestMode
			Confirm                       = $false
			SafeModeAdministratorPassword = (ConvertTo-SecureString -AsPlainText -String $using:SafeModePassword -Force)
			WarningAction                 = 'Ignore'
		}
		$null = Install-ADDSForest @forestParams
	}
}

function Test-PowerLabActiveDirectoryForest {
	param(
		[Parameter(Mandatory)]
		[pscredential]$Credential,

		[Parameter()]
		[string]$VMName = 'LABDC'
	)

	Invoke-Command -Credential $Credential -ScriptBlock { Get-AdUser -Filter * }
}

function New-PowerLabServer {
	[CmdletBinding(DefaultParameterSetName = 'Generic')]
	param
	(
		[Parameter(Mandatory)]
		[string]$Name,

		[Parameter(Mandatory)]
		[pscredential]$DomainCredential,

		[Parameter(Mandatory)]
		[pscredential]$VMCredential,

		[Parameter()]
		[string]$VMPath = 'C:\PowerLab\VMs',

		[Parameter()]
		[int64]$Memory = 4GB,

		[Parameter()]
		[string]$Switch = 'PowerLab',

		[Parameter()]
		[int]$Generation = 2,

		[Parameter()]
		[string]$DomainName = 'powerlab.local',

		[Parameter()]
		[ValidateSet('SQL', 'Web')]
		[string]$ServerType,

		[Parameter(ParameterSetName = 'SQL')]
		[ValidateNotNullOrEmpty()]
		[string]$AnswerFilePath = "C:\Program Files\WindowsPowerShell\Modules\PowerLab\SqlServer.ini",

		[Parameter(ParameterSetName = 'Web')]
		[switch]$NoDefaultWebsite
	)

	## Tworzenie maszyny wirtualnej
	$vmparams = @{
		Name       = $Name
		Path       = $VmPath
		Memory     = $Memory
		Switch     = $Switch
		Generation = $Generation
	}
	New-PowerLabVm @vmParams

	Install-PowerLabOperatingSystem -VmName $Name
	Start-VM -Name $Name

	Wait-Server -Name $Name -Status Online -Credential $VMCredential

	$addParams = @{
		DomainName = $DomainName
		Credential = $DomainCredential
		Restart    = $true
		Force      = $true
	}
	Invoke-Command -VMName $Name -ScriptBlock { Add-Computer @using:addParams } -Credential $VMCredential

	Wait-Server -Name $Name -Status Offline -Credential $VMCredential

	Wait-Server -Name $Name -Status Online -Credential $DomainCredential

	if ($PSBoundParameters.ContainsKey('ServerType')) {
		switch ($ServerType) {
			'Web' {
				Write-Host 'Tworzenie serwerów web nie jest jeszcze obsługiwane.'
				break
			}
			'SQL' {
				$tempFile = Copy-Item -Path $AnswerFilePath -Destination "C:\Program Files\WindowsPowerShell\Modules\PowerLab\temp.ini" -PassThru
				Install-PowerLabSqlServer -ComputerName $Name -AnswerFilePath $tempFile.FullName -DomainCredential $DomainCredential
				break
			}
			default {
				throw "Nieznany typ serwera: [$_]"
			}
		}
	}
}

function Wait-Server {
	[CmdletBinding()]
	param(
		[Parameter(Mandatory)]
		[string]$Name,

		[Parameter(Mandatory)]
		[ValidateSet('Online', 'Offline')]
		[string]$Status,

		[Parameter(Mandatory)]
		[pscredential]$Credential
	)

	if ($Status -eq 'Online') {
		$scriptBlock = {Invoke-Command -VmName $Name -ScriptBlock { 1 } -Credential $Credential -ErrorAction Ignore}
	} elseif ($Status -eq 'Offline') {
		$scriptBlock = {(-not (Invoke-Command -VmName $Name -ScriptBlock { 1 } -Credential $Credential -ErrorAction Ignore))}
	}
	while (-not (& $scriptBlock)) {
		Start-Sleep -Seconds 10
		Write-Host "Oczekiwanie na uruchomienie serwera [$Name]..."
	}
}

function Install-PowerLabSqlServer {
	param
	(
		[Parameter(Mandatory)]
		[string]$ComputerName,

		[Parameter(Mandatory)]
		[pscredential]$DomainCredential,

		[Parameter(Mandatory)]
		[ValidateNotNullOrEmpty()]
		[string]$AnswerFilePath,

		[Parameter()]
		[ValidateNotNullOrEmpty()]
		[string]$IsoFilePath = 'C:\PowerLab\ISOs\en_sql_server_2016_standard_x64_dvd_8701871.iso'
	)

	try {
		## Tworzy sesję PowerShell Direct, aby skopiować pliki z hosta na maszynę wirtualną
		Write-Verbose -Message "Tworzenie nowej sesji PSSession z [$($ComputerName)]..."
		$session = New-PSSession -VMName $ComputerName -Credential $DomainCredential

		## Sprawdza, czy SQL Server jest już zainstalowany
		if (Invoke-Command -Session $session -ScriptBlock { Get-Service -Name 'MSSQLSERVER' -ErrorAction Ignore }) {
			Write-Verbose -Message 'SQL Server jest już zainstalowany'
		} else {

			PrepareSqlServerInstallConfigFile -Path $AnswerFilePath

			$copyParams = @{
				Path        = $AnswerFilePath
				Destination = 'C:\'
				ToSession   = $session
			}
			Copy-Item @copyParams
			Copy-Item -Path $IsoFilePath -Destination 'C:\' -Force -ToSession $session

			$icmParams = @{
				Session      = $session
				ArgumentList = $AnswerFilePath, $IsoFilePath
				ScriptBlock  = {
					$image = Mount-DiskImage -ImagePath $args[1] -PassThru
					$installerPath = "$(($image | Get-Volume).DriveLetter):"
					$null = & "$installerPath\setup.exe" "/CONFIGURATIONFILE=C:\$($args[0])"
					$image | Dismount-DiskImage
				}
			}
			Invoke-Command @icmParams

			$scriptBlock = { Remove-Item -Path $using:IsoFilePath, $using:AnswerFilePath -ErrorAction Ignore }
			Invoke-Command -ScriptBlock $scriptBlock -Session $session
		}
		$session | Remove-PSSession
	} catch {
		$PSCmdlet.ThrowTerminatingError($_)
	}
}

function PrepareSqlServerInstallConfigFile {
	[CmdletBinding()]
	param
	(
		[Parameter(Mandatory)]
		[string]$Path,

		[Parameter()]
		[string]$ServiceAccountName = 'PowerLabUser',

		[Parameter()]
		[string]$ServiceAccountPassword = 'P@$$w0rd12',

		[Parameter()]
		[string]$SysAdminAcountName = 'PowerLabUser'
	)

	$configContents = Get-Content -Path $Path -Raw
	$configContents = $configContents.Replace('SQLSVCACCOUNT=""', ('SQLSVCACCOUNT="{0}"' -f $ServiceAccountName))
	$configContents = $configContents.Replace('SQLSVCPASSWORD=""', ('SQLSVCPASSWORD="{0}"' -f $ServiceAccountPassword))
	$configContents = $configContents.Replace('SQLSYSADMINACCOUNTS=""', ('SQLSYSADMINACCOUNTS="{0}"' -f $SysAdminAcountName))
	Set-Content -Path $Path -Value $configContents
}